import { focusRing } from "@/lib/styles";
import { cn } from "@/lib/utils";
import { Check, Copy } from "lucide-react";
import { ReactNode, useId, useRef, useState } from "react";
import { Button } from "./button";
import { Input } from "./input";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "./tooltip";
import { ComponentWithClassName } from "@/types/utils";
import { Label } from "./label";

export function InputCopy({ className, label, defaultValue }: ComponentWithClassName<{ defaultValue: string | number | undefined | null, label?: string | ReactNode }>) {
  const [copied, setCopied] = useState<boolean>(false);
  const inputRef = useRef<HTMLInputElement>(null);
  const id = useId()

  const handleCopy = () => {
    if (inputRef.current) {
      navigator.clipboard.writeText(inputRef.current.value);
      setCopied(true);
      setTimeout(() => setCopied(false), 1500);
    }
  };

  return (
    <div className={cn("flex flex-col gap-1", className)}>
      {typeof label === 'string' || typeof label === "number" ? (
        <Label className="text-xs text-foreground" htmlFor={id}>{label}</Label>
      ) : (
        label
      )}

      <div className="relative flex flex-row">
        <Input
          id={id}
          ref={inputRef}
          className="pe-9 rounded-e-none"
          type="text"
          defaultValue={defaultValue ?? ""}
          readOnly
        />

        <TooltipProvider delayDuration={0}>
          <Tooltip>
            <TooltipTrigger asChild>
              <Button
                onClick={handleCopy}
                variant={"outline"}
                size="icon"
                className={cn(focusRing, "h-full rounded-none shadow-sm rounded-s-none rounded-e-lg border border-border border-l-transparent")}
                aria-label={copied ? "Copied" : "Copy to clipboard"}
                disabled={copied}
              >
                <div
                  className={cn(
                    "transition-all",
                    copied ? "scale-100 opacity-100" : "scale-0 opacity-0",
                  )}
                >
                  <Check
                    className="stroke-emerald-500 size-4"
                    aria-hidden="true"
                  />
                </div>
                <div
                  className={cn(
                    "absolute transition-all",
                    copied ? "scale-0 opacity-0" : "scale-100 opacity-100",
                  )}
                >
                  <Copy className="text-muted-foreground size-3" aria-hidden="true" />
                </div>
              </Button>
            </TooltipTrigger>
            <TooltipContent className="border border-input bg-popover px-2 py-1 text-xs text-muted-foreground">
              Copy to clipboard
            </TooltipContent>
          </Tooltip>
        </TooltipProvider>
      </div>
    </div>
  );
}

